/***************************************************************************
 **************** Hochschule Technik + Architektur Luzern ******************
 ****************       Fachbereich Computertechnik       ******************
 ***************************************************************************
 * Projekt:                   
 * Filename:                  LCDrv_PC.C
 * Autor:                     P. Rsli   / R
 * Compiler, Language:        HiCross V2.7  C
 * Target:                    CC11 mit Motorola 68HC11,
 *                            MEDUSA-Trainer, Pinboardadapter mit Keyboard
 *--------------------------------------------------------------------------
 * Aenderungen:
 *
 *   Version  Datum     Autor  Beschreibung der Aenderung
 *
 *   1.0      24.04.98   R    Erstellung
 *   1.1      02.05.99   Kp    div. Anpassungen
 *   1.2      03.05.00   Kp    Prefix in Interfacefunktionsnamen eingefgt
 *   1.3      08.02.01   M    Treiber umgeschrieben fuer Betrieb am PORTC
 *                             (DataReg) und PORTB (CtrlReg) des HC11
 *   1.4      03.03.01   Kp    kleine Anpassungen
 *   1.5      05.03.01   Kp    mehr Sicherheit beim Umschalten des Datenbus
 *   1.6	  03.03.02	 Kp	   Anpassungen an neues HC11F1.H         
 * 
 *
 *--------------------------------------------------------------------------
 * Kurzbeschreibung:
 *   Diese Datei stellt Routinen fr die Ansteuerung des 2x16 Charakter
 *   LC-Displays (LCD-Controller) HD44780 zur Verfgung. 
 *   Das Display wird im Zweizeilenmodus initialisiert.Fr eine genaue 
 *   Beschreibung des Displays siehe Datenbltter.
 *
 *	 Anschluss des Displays an folgenden HC-11 Ports:
 *		Datenbus am PORTC (bidirektionaler Betrieb)
 *		Controlbus:	RS  an PORTB.PB0
 *					R_W an PORTB.PB1
 *					EN  an PORTB.PB2
 *
 *	 Systemvoraussetzungen: CC-11 mit bestckter PRU   
 ***************************************************************************/

/***************************************************************************
   INCLUDES
 ****************************************************************************/       
#include "LCDrv_PC.H"
#include<hc11f1.h>     // HC-11-F1 Registerdefinitionen


/***************************************************************************
   TYPES
 ****************************************************************************/       

/***************************************************************************
   DEFINES
 ****************************************************************************/       

/* HW-Registermapping */
#define CtrlReg PORTB_	// ControlSignals an PORTB[0..2]
#define RS  PB0
#define R_W PB1
#define EN  PB2
#define DataReg PORTC	// Databus an PORTC
#define DataDir DDRC
#define Input 0			// DataDirection = Input
#define Output 0xFF		// DataDirection = Output


/* DEFINES fr Display-Funktionen */
#define TwoLine5x7Dot  0x38
#define ClearDisplay   0x01
#define ReturnHome     0x02
#define SecondLine     0xC0
#define NoShift        0x06
#define RightShift     0x1C
#define LeftShift      0x18
#define GotoXY         0x80
#define CursorOn       0x0E
#define CursorOff      0x0C


/***************************************************************************
   private Funktionen:
   lowlevel Buszugriffsroutinen
   
   Implementationshinweise:
   - >>> WICHTIG: unbedingt Buskurzschluss vermeiden, 
     d.h. nie gleichzeitig CtrlReg.R_W = 0 und DDRC auf Output
   - Zugriffszyklus gemss Spezifikation mit den Steuersignalen nachbilden
   - wo ntig, mit einem NOP (asm NOP) verlngern
 ****************************************************************************/       
/***************************************************************************/
unsigned char ReadLCDData ()
/***************************************************************************
   Funktion:    liest das LCD-Datenregister.
   Parameter:
     In         -
     Out        RETURN  : Gelesenes LCD-Zeichen
     InOut      -            
   *************************************************************************/
{
   unsigned char Ch;
   DataDir = Input;                             /* Datenregister auf Input */
   CtrlReg.EN = CtrlReg.R_W = CtrlReg.RS = 0;   /* alle Steuersignale =0 */
   CtrlReg.R_W = CtrlReg.RS = 1;         /* R_W = 1: Read, RS = 1: Data-Reg.*/
   CtrlReg.EN = 1;                       /* EN = 1: Enable Data-Read */
   asm NOP;                              /* wait a bit */
   Ch = DataReg;                         /* Daten lesen */  
   CtrlReg.EN = 0;                       /* EN =0 */
   CtrlReg.R_W = 0;                      /* LCD-Dataport auf Write */
   DataDir = Input;                      /* HC11-Dataport auf Read */
   return Ch;
}

/***************************************************************************/
unsigned char ReadLCDStatus ()
/***************************************************************************
   Funktion:    liest das LCD-Statusregister.
   Parameter:
     In         -
     Out        RETURN  : Gelesener LCD-Statuswert
     InOut      -            
   *************************************************************************/
{
   unsigned char Ch;
   DataDir = Input;
   CtrlReg.EN = CtrlReg.R_W = CtrlReg.RS = 0;    /* alle Steuersignale =0 */
   CtrlReg.R_W = 1;                     /* R_W = 1: Read */
   CtrlReg.EN = 1;                      /* EN = 1: Enable Data-Read */
   asm NOP;                             /* wait a bit */
   Ch = DataReg;                        /* Status lesen */  
   CtrlReg.EN = 0;                      /* EN =0 */
   CtrlReg.R_W = 0;                     /* LCD-Dataport auf Write */
   DataDir = Input;                     /* HC11-Dataport auf Read */
   return Ch;
}

/***************************************************************************/
void WaitForLCDReady ()
/***************************************************************************
   Funktion:    wartet bis das LCD zur Datenaufnahme bereit
                ist (BusyFlag (MSB) im Status-Register gelscht)
   Parameter:
     In         -
     Out        -
     InOut      -            
   *************************************************************************/
{
   while (ReadLCDStatus() >= 0x80);
}

/***************************************************************************/
void WriteLCDData (unsigned char Ch)
/***************************************************************************
   Funktion:    schreibt ins LCD-Datenregister,
                wenn das LCD bereit ist.
   Parameter:
     In         Ch      : zu schreibendes Zeichen
     Out        -
     InOut      -            
   *************************************************************************/
{
   WaitForLCDReady();                   /* Wartet, bis LCD bereit ist */
   CtrlReg.EN = CtrlReg.R_W = CtrlReg.RS = 0;    /* alle Steuersignale =0 */
   DataDir = Output;                    /* HC11-Dataport auf Write */
   CtrlReg.RS = 1;                      /* RS = 1: select Data-Reg */
   CtrlReg.EN = 1;                      /* Enable = 1 */
   DataReg = Ch;                        /* Daten anlegen */
   asm NOP;                             /* wait a bit */
   CtrlReg.EN = 0;                      /* Enable = 0: Daten uebernehmen */
   DataDir = Input;                     /* HC11-Dataport auf Read */
   CtrlReg.R_W = 0;                     /* LCD-Dataport auf Write */
}

/***************************************************************************/
void WriteLCDCommand (unsigned char Ch)
/***************************************************************************
   Funktion:    schreibt ins LCD-Commandregister,
                wenn das LCD bereit ist.
   Parameter:
     In         Ch      : zu schreibender Command-Code
     Out        -
     InOut      -            
   *************************************************************************/
{
   WaitForLCDReady();                   /* Wartet, bis LCD bereit ist */
   CtrlReg.EN = CtrlReg.R_W = CtrlReg.RS = 0;    /* alle Steuersignale =0 */
   DataDir = Output;                    /* HC11-Dataport auf Write */
   CtrlReg.EN = 1;                      /* EN-Signal setzen */
   DataReg = Ch;                        /* Kommando anlegen */
   asm NOP;                             /* wait a bit */
   CtrlReg.EN = 0;                      /* Enable = 0: Daten uebernehmen */
   DataDir = Input;                     /* HC11-Dataport auf Read */
   CtrlReg.R_W = 0;                     /* LCD-Dataport auf Write */
}



/***************************************************************************/
void LCDInit()
/***************************************************************************
   Funktion:    initialisiert das LCD.
                (gem. Hersteller Spezifikation)
   Parameter:
     In         -
     Out        -
     InOut      -            
   *************************************************************************/
{
   unsigned int i;
   DataDir = Input;                     
   CtrlReg.EN = CtrlReg.R_W = CtrlReg.RS = 0;         /* alle Steuersignale =0 */
   DataDir = Output;
   for (i=0; i<10000; i++);             /* warte lnger als 30ms */
   DataReg = TwoLine5x7Dot;             /* Function set: 2-line mode, 5 x 7 dots */
   CtrlReg.EN = 1;                      /* EN-Impuls */
   asm NOP;                             /* wait a bit */
   CtrlReg.EN = 0;            
   for (i=0; i<10000; i++);             /* warte lnger als 39ms */
   DataReg = 0x0E;                      /* Disp On/Off Ctrl: display on, cursor on, no blink */
   CtrlReg.EN = 1;                      /* EN-Impuls */
   asm NOP;                             /* wait a bit */
   CtrlReg.EN = 0;            
   for (i=0; i<10000; i++);             /* warte lnger als 39ms */
   DataReg = 0x01;                      /* clear display */
   CtrlReg.EN = 1;                      /* EN-Impuls */
   asm NOP;                             /* wait a bit */
   CtrlReg.EN = 0;            
   for (i=0; i<1000; i++);              /* warte lnger als 1.53ms */
   DataReg = 0x06;                      /* Entry mode set: increment mode, disp shift off */
   CtrlReg.EN = 1;                      /* EN-Impuls */
   asm NOP;                             /* wait a bit */
   CtrlReg.EN = 0;
   CtrlReg.R_W = 0;                     /* LCD-Dataport auf Write */
   DataDir = Input;                     /* HC11-Dataport auf Read */
   
   WriteLCDCommand (TwoLine5x7Dot);     /* ist zusaetzlich noetig (Bug?) */
}

/***************************************************************************/
void LCDClear()
/***************************************************************************
   Funktion:    lscht das Display und setzt den Cursor auf
                den Anfang der ersten Zeile.
   
   Parameter:
     In         -
     Out        -
     InOut      -            
   *************************************************************************/
{
   WriteLCDCommand (ClearDisplay);
}

/***************************************************************************/
void LCDHome()
/***************************************************************************
   Funktion:    Setzt den Cursor auf den Anfang der ersten Zeile
   Parameter:
     In         -
     Out        -
     InOut      -   
   *************************************************************************/
{
   WriteLCDCommand (ReturnHome);
}

/***************************************************************************/
void LCDCursor (unsigned char On)
/***************************************************************************
   Funktion:    schaltet den Cursor ein (On != 0) oder aus (On = 0). 
                
   Parameter:
     In         On      : On=0 -> Cursor aus , On!=0 -> Cursor ein
     Out        -
     InOut      -   
   *************************************************************************/
{
   if (On)
      WriteLCDCommand (CursorOn);
   else
      WriteLCDCommand (CursorOff);
}

/***************************************************************************/
void LCDGoto (unsigned char X, unsigned char Y)
/***************************************************************************
   Funktion:    setzt den Cursor an die angegebene Position.
   Parameter:
     In         X       : X-Position auf dem Display
                Y       : Y-Position auf dem Display
     Out        -
     InOut      -   
   *************************************************************************/
{
   WriteLCDCommand (GotoXY + X + Y * 0x40);
}

/***************************************************************************/
void LCDShift (unsigned char Right)
/***************************************************************************
   Funktion:    Weil eine Zeile 40 Zeichen lang sein kann, davon aber nur
                16 Zeichen angezeigt werden, kann mit dieser Funktion der
                sichtbare Bereich nach rechts (bei Right!=0) oder links (bei
                Right=0) geschoben werden. Die aktuelle Cursorposition wird
                dabei nicht verndert.
   Parameter:
     In         Right   : =0 -> links schieben;  !=0 -> rechts schieben
     Out        -
     InOut      -   
   *************************************************************************/
{
   if (Right)
      WriteLCDCommand (RightShift);
   else
      WriteLCDCommand (LeftShift);
}

/***************************************************************************/
void LCDWrite (char Ch)
/***************************************************************************
   Funktion:    gibt das Zeichen Ch an der aktuellen Cursorposition 
                des Displays aus.
   Parameter:
     In         Ch      : auszugebendes Zeichen
     Out        -
     InOut      -   
   *************************************************************************/
{
   WriteLCDData (Ch);
}

/***************************************************************************/
void LCDWriteLn()
/***************************************************************************
   Funktion:    Setzt den Cursor auf den Anfang der nchsten Zeile.    
   Parameter:
     In         -
     Out        -
     InOut      -   
   *************************************************************************/
{
   WriteLCDCommand (SecondLine);
}

/***************************************************************************/
void LCDWriteString (char Str[])
/***************************************************************************
   Funktion:    gibt die Zeichenkette Str (\0-terminiert) ab
                der aktuellen Cursorposition auf das Display aus.
   Parameter:
     In         Str     : Zeiger auf eine nullterminierte Zeichenkette
     Out        -
     InOut      -   
   *************************************************************************/
{
   unsigned char i=0;
   while (Str[i] != '\0'){
      LCDWrite (Str[i++]);
   }
}


/***************************************************************************/
void LCDLoadSoftChar (unsigned char CharCode, unsigned char SoftChar[])
/***************************************************************************
   Funktion:    ldt ein anwenderspezifisches Zeichen (Soft-Charakter 
                5 x 8 Dots) in das CG-RAM des Displays.
                Das Zeichen kann danach wie ein vordefiniertes Zeichen mit
                seinem CharCode benutzt werden.
   Parameter:
     In         CharCode:   Code des Spezialzeichens (0..7)
                SoftChar:   Definition des Spezialzeichens
     Out        -
     InOut      -   
   *************************************************************************/
{
         unsigned char CGAdress = 0x40; // Basisadresse fr die Softchar
   const unsigned char DDAdress = 0x80; // Beenden des Downloadsmodus
         unsigned char i;
        
   // CG-Ram setzen
   CGAdress += CharCode * 0x08;  // Zeichencode * 0x08 [Lnge]
   WriteLCDCommand(CGAdress);
   
   for (i=0; i<=7; i++){
     WriteLCDData(SoftChar[i]);
   }
   // Download beenden
   WriteLCDCommand(DDAdress);
}
